﻿using Basler.Pylon;
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Diagnostics;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Windows.Forms;
using HalconDotNet;
using System.Runtime.InteropServices;

namespace Pylon_Halcon
{
    public partial class Main : Form
    {
        private PixelDataConverter converter = new PixelDataConverter();
        private Stopwatch stopWatch = new Stopwatch();
        Camera camera = null;
        private IntPtr BufferAdr = IntPtr.Zero;
        private HObject _RGBImage = null;
        private HImage img = null;
        public Main()
        {
            InitializeComponent();
            // Place the pointer to the buffer of the bitmap.
            converter.OutputPixelFormat = PixelType.BGRA8packed;
            InitCamera();
        }

        private void InitCamera()
        {
            if (camera == null)
            {
                camera = new Camera();
                // Set the acquisition mode to free running continuous acquisition when the camera is opened.
                camera.CameraOpened += Configuration.AcquireContinuous;
                camera.StreamGrabber.ImageGrabbed += OnImageGrabbed;
                // Open the connection to the camera device.
                camera.Open();

                // prepare the converter
                converter.OutputPixelFormat = PixelType.RGB8packed;
                converter.Parameters[PLPixelDataConverter.InconvertibleEdgeHandling].SetValue("Clip");

                string oldPixelFormat = camera.Parameters[PLCamera.PixelFormat].GetValue();

                IEnumerable<String>  Values = camera.Parameters[PLCamera.PixelFormat].GetAllValues();
                List<String> AviableValues = new List<string>();
                foreach (string val in Values)
                {
                    if (camera.Parameters[PLCamera.PixelFormat].TrySetValue(val))
                        ComboPixelFormat.Items.Add(val);
                }
                int pos = ComboPixelFormat.FindString(oldPixelFormat);
                if(pos > -1)
                {
                    ComboPixelFormat.SelectedIndex = pos;
                    camera.Parameters[PLCamera.PixelFormat].TrySetValue(oldPixelFormat);
                }

                
            }

        }

        private void BtnSingleGrab_Click(object sender, EventArgs e)
        {
            try
            {
                if (camera != null && !camera.StreamGrabber.IsGrabbing)
                {
                    // The parameter MaxNumBuffer can be used to control the amount of buffers
                    // allocated for grabbing. The default value of this parameter is 10.
                    camera.Parameters[PLCameraInstance.MaxNumBuffer].SetValue(1);
                    // Start grabbing.
                    camera.StreamGrabber.Start(1, GrabStrategy.LatestImages, GrabLoop.ProvidedByStreamGrabber);
                }
            }
            catch (Exception ex)
            {
                ShowException(ex);
            }
            finally
            {

            }
        }
        // Occurs when an image has been acquired and is ready to be processed.
        private void OnImageGrabbed(Object sender, ImageGrabbedEventArgs e)
        {
            if (InvokeRequired)
            {
                // If called from a different thread, we must use the Invoke method to marshal the call to the proper GUI thread.
                // The grab result will be disposed after the event call. Clone the event arguments for marshaling to the GUI thread.
                BeginInvoke(new EventHandler<ImageGrabbedEventArgs>(OnImageGrabbed), sender, e.Clone());
                return;
            }
            try
            {
                // Acquire the image from the camera. Only show the latest image. The camera may acquire images faster than the images can be displayed.

                // Get the grab result.
                IGrabResult grabResult = e.GrabResult;

                // Check if the image can be displayed.
                if (grabResult.IsValid)
                {
                    // Reduce the number of displayed images to a reasonable amount if the camera is acquiring images very fast.
                    if (!stopWatch.IsRunning || stopWatch.ElapsedMilliseconds > 33)
                    {
                        stopWatch.Restart();
                        ShowMesage("Current Image ID : " + grabResult.ImageNumber, Color.Green);

                        if (grabResult.PixelTypeValue == PixelType.Mono8)
                        {

                            // Access the image data.
                            byte[] buffer = grabResult.PixelData as byte[];

                            GCHandle pinnedArray = GCHandle.Alloc(buffer, GCHandleType.Pinned);
                            IntPtr ptrArray = pinnedArray.AddrOfPinnedObject();

                            HImage img = new HImage("byte", grabResult.Width, grabResult.Height, ptrArray);
                            HImageWindow.HalconWindow.AttachBackgroundToWindow(img);
                            pinnedArray.Free();
                        }
                        else if (grabResult.PixelTypeValue == PixelType.BayerBG8 || grabResult.PixelTypeValue == PixelType.BayerGB8
                                 || grabResult.PixelTypeValue == PixelType.BayerRG8 || grabResult.PixelTypeValue == PixelType.YUV422packed
                                  )
                        {

                            //allocate the m_stream_size amount of bytes in non-managed environment 
                            if (BufferAdr == IntPtr.Zero)
                            {
                                BufferAdr = Marshal.AllocHGlobal((Int32)converter.GetBufferSizeForConversion(grabResult));
                            }
                            converter.Convert(BufferAdr, converter.GetBufferSizeForConversion(grabResult), grabResult);

                            if (img == null)
                                img = new HImage("byte", grabResult.Width, grabResult.Height);

                            //update the image with current pixel data
                            img.GenImageInterleaved(BufferAdr, "rgb", (HTuple)grabResult.Width - 1, (HTuple)grabResult.Height - 1, -1,
                                                            "byte", (HTuple)grabResult.Width - 1, (HTuple)grabResult.Height
                                                            - 1, 0, 0, -1, 0);
                            //display the image
                            HImageWindow.HalconWindow.AttachBackgroundToWindow(img);

                        }
                        else if (grabResult.PixelTypeValue == PixelType.RGB8packed)
                        {
                            if (img == null)
                                img = new HImage("byte", grabResult.Width, grabResult.Height);

                            // Access pylon the image data.
                            byte[] buffer = grabResult.PixelData as byte[];

                            GCHandle pinnedArray = GCHandle.Alloc(buffer, GCHandleType.Pinned);
                            IntPtr ptrArray = pinnedArray.AddrOfPinnedObject();

                            //update the image with current pixel data
                            img.GenImageInterleaved(ptrArray, "rgb", (HTuple)grabResult.Width, (HTuple)grabResult.Height, -1,
                                                    "byte", (HTuple)grabResult.Width, (HTuple)grabResult.Height, 0, 0, -1, 0);
                            //display the image
                            HImageWindow.HalconWindow.AttachBackgroundToWindow(img);
                            pinnedArray.Free();
                        }
                        else if(grabResult.PixelTypeValue == PixelType.BGR8packed)
                        {
                            if (img == null)
                                img = new HImage("byte", grabResult.Width, grabResult.Height);

                            // Access pylon the image data.
                            byte[] buffer = grabResult.PixelData as byte[];

                            GCHandle pinnedArray = GCHandle.Alloc(buffer, GCHandleType.Pinned);
                            IntPtr ptrArray = pinnedArray.AddrOfPinnedObject();

                            //update the image with current pixel data
                            img.GenImageInterleaved(ptrArray, "bgr", (HTuple)grabResult.Width, (HTuple)grabResult.Height, -1,
                                                    "byte", (HTuple)grabResult.Width, (HTuple)grabResult.Height, 0, 0, -1, 0);
                            //display the image
                            HImageWindow.HalconWindow.AttachBackgroundToWindow(img);
                            pinnedArray.Free();

                        }
                        else
                        {
                            ShowMesage("Unknown pixel format", Color.Red);
                        }

                    }
             
                }
            }
            catch (Exception exception)
            {
                ShowException(exception);
            }
            finally
            {
                // Dispose the grab result if needed for returning it to the grab loop.
                e.DisposeGrabResultIfClone();
            }
        }
        private void ShowMesage(string msg, Color cl)
        {
            // to be done : clean up the list
            if (InvokeRequired)
            {
                // If called from a different thread, we must use the Invoke method to marshal the call to the proper thread.
                BeginInvoke(new Action(() => this.ShowMesage(msg, cl)));
                return;
            }

            if (LogBox.TextLength > 3000)
                LogBox.Clear();
            LogBox.SelectionStart = LogBox.Text.Length;
            LogBox.ScrollToCaret();

            if (!msg.Contains((char)13))
                msg += "\n";

            LogBox.SelectionColor = cl;
            LogBox.AppendText(DateTime.Now.ToString("HH:mm:ss : ") + " : " + msg);
            LogBox.SelectionStart = LogBox.Text.Length;
            LogBox.ScrollToCaret();
            LogBox.ForeColor = LogBox.ForeColor;

        }
        private void ShowException(Exception exception)
        {
            StackTrace stackTrace = new StackTrace();
            MessageBox.Show("Exception caught:\n" + exception.Message + "  called from " + stackTrace.GetFrame(1).GetMethod().Name, "Error", MessageBoxButtons.OK, MessageBoxIcon.Error);
            //  MessageBox.Show("Exception caught:\n" + exception.Message, "Error", MessageBoxButtons.OK, MessageBoxIcon.Error);
        }

        private void BtnGrabContinues_Click(object sender, EventArgs e)
        {
            try
            {
                if (camera != null && !camera.StreamGrabber.IsGrabbing)
                {
                    // allocated for grabbing. The default value of this parameter is 10.
                    camera.Parameters[PLCameraInstance.MaxNumBuffer].SetValue(1);
                    // Start the grabbing of images until grabbing is stopped.
                    camera.Parameters[PLCamera.AcquisitionMode].SetValue(PLCamera.AcquisitionMode.Continuous);
                    camera.StreamGrabber.Start(GrabStrategy.LatestImages, GrabLoop.ProvidedByStreamGrabber);

                }
            }
            catch (Exception exception)
            {
                ShowException(exception);
            }
        }

        private void BtnStop_Click(object sender, EventArgs e)
        {
            camera.StreamGrabber.Stop();
        }

        private void BtnExit_Click(object sender, EventArgs e)
        {
            Application.Exit();
        }
        private void ComboPixelFormat_SelectedIndexChanged(object sender, EventArgs e)
        {
            string val = ComboPixelFormat.SelectedItem.ToString();
            if(camera != null && !camera.StreamGrabber.IsGrabbing)
            {
                camera.Parameters[PLCamera.PixelFormat].TrySetValue(val);
            }
        }

        // Closes the camera object and handles exceptions.
        private void DestroyCamera()
        {
            // Disable all parameter controls.
            try
            {
                if (camera != null && camera.StreamGrabber.IsGrabbing)
                {
                    camera.StreamGrabber.Start();
                }
            }
            catch (Exception exception)
            {
                ShowException(exception);
            }

            // Destroy the camera object.
            try
            {
                if (camera != null)
                {
                    camera.Close();
                    camera.Dispose();
                    camera = null;
                }
            }
            catch (Exception exception)
            {
                ShowException(exception);
            }

            // Destroy the converter object.
            if (converter != null)
            {
                converter.Dispose();
                converter = null;
            }
        }

        private void Main_FormClosed(object sender, FormClosedEventArgs e)
        {
            DestroyCamera();
        }
    }
}
