// GenericParameterAccess_PixelFormat.cpp
/*
    Note: Before getting started, Basler recommends reading the "Programmer's Guide" topic
    in the pylon C++ API documentation delivered with pylon.
    If you are upgrading to a higher major version of pylon, Basler also
    strongly recommends reading the "Migrating from Previous Versions" topic in the pylon C++ API documentation.

    For camera configuration and for accessing other parameters, the pylon API
    uses the technologies defined by the GenICam standard hosted by the
    European Machine Vision Association (EMVA). The GenICam specification
    (http://www.GenICam.org) defines a format for camera description files.
    These files describe the configuration interface of GenICam compliant cameras.
    The description files are written in XML (eXtensible Markup Language) and
    describe camera registers, their interdependencies, and all other
    information needed to access high-level features such as Gain,
    Exposure Time, or Image Format by means of low-level register read and
    write operations.

    The elements of a camera description file are represented as software
    objects called Nodes. For example, a node can represent a single camera
    register, a camera parameter such as Gain, a set of available parameter
    values, etc. Each node implements the GenApi::INode interface.

    The nodes are linked together by different relationships as explained in the
    GenICam standard document available at www.GenICam.org. The complete set of
    nodes is stored in a data structure called Node Map.
    At runtime, a Node Map is instantiated from an XML description.

    This sample shows the 'generic' approach for configuring a camera
    using the GenApi nodemaps represented by the GenApi::INodeMap interface.

    The names and types of the parameter nodes can be found in the Basler pylon Programmer's Guide
    and API Reference, in the camera User's Manual, in the camera's document about
    Register Structure and Access Methodes (if applicable), and by using the pylon Viewer tool.

    See also the ParametrizeCamera_NativeParameterAccess sample for the 'native'
    approach for configuring a camera.
*/

// Include files to use the pylon API.
#include <pylon/PylonIncludes.h>

// Namespace for using pylon objects.
using namespace Pylon;

// Namespace for using GenApi objects.
using namespace GenApi;

// Namespace for using cout.
using namespace std;

using namespace GENICAM_NAMESPACE;

String_t GetAccessModeAsString(EAccessMode accessMode)
{
	gcstring s;
	switch (accessMode)
	{
	case EAccessMode::NA:
		s.assign("NA");
		break;
	case EAccessMode::NI:
		s.assign("NI");
		break;
	case EAccessMode::RO:
		s.assign("RO");
		break;
	case EAccessMode::RW:
		s.assign("RW");
		break;
	case EAccessMode::WO:
		s.assign("WO");
		break;
	case EAccessMode::_CycleDetectAccesMode:
		s.assign("_CycleDetectAccesMode");
		break;
	case EAccessMode::_UndefinedAccesMode:
	default:
		s.assign("_UndefinedAccesMode");
		break;
	}
	return s;
}



int main(int argc, char* argv[])
{

    // The exit code of the sample application.

    int exitCode = 0;

    // Before using any pylon methods, the pylon runtime must be initialized. 
    PylonInitialize();

    try
    {
		cout << "GenericParameterAccess_PixelFormat" << endl
		     << "==================================" << endl << endl;

        // Create an instant camera object with the camera found first.
		cout << "Create an Pylon::CInstantCamera object with the camera found first." << endl
			<< "    CInstantCamera camera(CTlFactory::GetInstance().CreateFirstDevice());" << endl << endl;
        CInstantCamera camera(CTlFactory::GetInstance().CreateFirstDevice());

		cout << "Retrieve the CDeviceInfo object from the camera." << endl
			<< "    CDeviceInfo device_info = camera.GetDeviceInfo();" << endl << endl;
		CDeviceInfo device_info = camera.GetDeviceInfo();

		// Get camera device information.
        cout << "        CDeviceInfo::GetFullName()    : " << device_info.GetVendorName() << endl;
		cout << "        CDeviceInfo::GetModelName()   : " << device_info.GetModelName() << endl;
		cout << "        CDeviceInfo::GetSerialNumber(): " << device_info.GetSerialNumber() << endl << endl;

		cout << "Retrieve the cameras nodemap for generic camera parameter access." << endl
			<< "    INodeMap& nodemap = camera.GetNodeMap();" << endl << endl;
		INodeMap& nodemap = camera.GetNodeMap();

		// Open the camera for accessing the parameters.
		cout << "Open the camera for accessing the parameters." << endl
			<< "    camera.Open();" << endl << endl;
		camera.Open();

		cout << "Create a smart pointer for <GENICAM_NAMESPACE::IEnumeration> and retrieve node \"PixelFormat\" from nodemap." << endl
			<< "    CEnumerationPtr ptrPixelFormat = nodemap.GetNode(\"PixelFormat\");" << endl << endl;
		CEnumerationPtr ptrPixelFormat = nodemap.GetNode("PixelFormat");

		cout << "Get current PixelFormat value as symbolic name." << endl
			<< "    String_t strPixelFormatSymbolic = ptrPixelFormat->ToString();" << endl;
		String_t strPixelFormatSymbolic = ptrPixelFormat->ToString();
		cout << "       Current PixelFormat as symbolic name: \"" << strPixelFormatSymbolic << "\"" << endl << endl;

		cout << "Retrieve current PixelFormat value as int64_t value." << endl
			<< "    int64_t intValuePixelFormat = ptrPixelFormat->GetIntValue();" << endl;
		int64_t intValuePixelFormat = ptrPixelFormat->GetIntValue();
		cout << "       Current PixelFormat as int64_t value in hexadecimal: 0x" << hex << intValuePixelFormat << endl << endl;


		cout << "Lists:" << endl << endl;

		cout << "Get list of symbolic values (implemented entries ONLY)." << endl
			<< "    ptrPixelFormat->GetSymbolics(StringList_t strListPixelFormats);" << endl;
		StringList_t strListPixelFormats;
		ptrPixelFormat->GetSymbolics(strListPixelFormats);
		for (StringList_t::iterator it = strListPixelFormats.begin(); it != strListPixelFormats.end(); it++)
		{
			String_t strSymbolic = *it;
			cout << "        " << strSymbolic.c_str() << endl;
		}
		cout << endl << endl;

		cout << "Get list of entry nodes (ALL entries from XML file, regardless if they're actually implemented or not)." << endl
			<< "    ptrPixelFormat->GetEntries(NodeList_t nodeListPixelFormats);" << endl << endl;

		NodeList_t nodeListPixelFormats;
		ptrPixelFormat->GetEntries(nodeListPixelFormats);
		gcstring_vector listName;
		gcstring_vector listSymbolic;
		std::vector<int64_t> listValue;
		int maxLenName = 0;
		int maxLenSymbolic = 0;
		for (NodeList_t::iterator it = nodeListPixelFormats.begin(); it != nodeListPixelFormats.end(); it++)
		{
			INode* nodeEntry = *it;
			
			// We want interface type IEnumEntry only
//			assert(nodeEntry->GetPrincipalInterfaceType() == intfIEnumEntry);
			if (nodeEntry->GetPrincipalInterfaceType() == intfIEnumEntry)
			{
				CEnumEntryPtr ptr(nodeEntry);
				gcstring name = nodeEntry->GetName();
				gcstring symbolic = ptr->GetSymbolic();
				int64_t value = ptr->GetValue();

				if (strlen(name.c_str()) > maxLenName)
					maxLenName = (int)strlen(name.c_str());
				if (strlen(symbolic.c_str()) > maxLenSymbolic)
					maxLenSymbolic = (int)strlen(symbolic.c_str());

				listName.push_back(name);
				listSymbolic.push_back(symbolic);
				listValue.push_back(value);
			}
		}

		gcstring_vector::iterator itName = listName.begin();
		gcstring_vector::iterator itSymbolic = listSymbolic.begin();
		std::vector<int64_t>::iterator itValue = listValue.begin();
		gcstring str;
		str.append("Name");
		for (int i = 0; i <= (maxLenName + 2 - strlen("Name")); i++)
			str += " ";
		str.append("Symbolic");
		for (int i = 0; i <= (maxLenSymbolic + 2 - strlen("Symbolic")); i++)
			str += " ";
		str.append("Value      ptrPixelFormat->GetEntry(intValue)->GetSymbolic()");
		cout << str << endl;

		for (itName; itName != listName.end();)
		{
			str.assign("");
			str.append(*itName);
			for (int i = 0; i <= (maxLenName + 2 - (*itName).length()); i++)
				str += " ";

			str.append(*itSymbolic);
			for (int i = 0; i <= (maxLenSymbolic + 2 - (*itSymbolic).length()); i++)
				str += " ";

			cout << str << "0x" << hex << *itValue << "  " << ptrPixelFormat->GetEntry(*itValue)->GetSymbolic() << endl;

			itName++;
			itSymbolic++;
			itValue++;
		}
		cout << endl << endl;
		
		cout << "EPixelType and CPixelTypeMapper:" << endl << endl;

		EPixelType pixeltype_from_name = CPixelTypeMapper::GetPylonPixelTypeByName(strPixelFormatSymbolic);
		cout << endl << "Pylon::EPixelType from static function Pylon::CPixelTypeMapper::GetPylonPixelTypeByName(String_t symbolicName)" << endl
			<< "    symbolicName: \"" << strPixelFormatSymbolic << "\"" << endl
			<< "    EPixelType  : 0x" << pixeltype_from_name << endl;

		cout << endl << "Instantiating CPixelTypeMapper(IEnumeration* pEnum)" << endl
			<< "    CPixelTypeMapper pixelTypeMapper(reinterpret_cast<IEnumeration *>(ptrPixelFormat->GetNode()));" << endl;
		CPixelTypeMapper pixelTypeMapper(reinterpret_cast<IEnumeration *>(ptrPixelFormat->GetNode()));

		EPixelType pixeltype_from_intvalue = pixelTypeMapper.GetPylonPixelTypeFromNodeValue(intValuePixelFormat);
		cout << endl << "EPixelType from CPixelTypeMapper::GetPylonPixelTypeFromNodeValue(int64_t intValue)" << endl
			<< "    intValue  : 0x" << intValuePixelFormat << endl
			<< "    EPixelType: 0x" << pixeltype_from_intvalue << endl;

		SFNCVersion sfnc_version;
		if (camera.GetSfncVersion() >= Sfnc_2_0_0)
			sfnc_version = SFNCVersion_2_0;
		else
			sfnc_version = SFNCVersion_pre2_0;

		cout << "GenICam symbolic name from CPixelTypeMapper::GetNameByPixelType(EPixelType enum_value, SFNCVersion version)" << endl;
		String_t strNameFromPixelType = pixelTypeMapper.GetNameByPixelType(pixeltype_from_intvalue, sfnc_version);
		if (strNameFromPixelType.empty())
		{
			cout << "    Name: returned string is empty.";
		}
		else
		{
			cout << "    Name: \"" << strNameFromPixelType << "\"" << endl << endl;
		}

		cout << "Get IEnumEntry node for \"" << strNameFromPixelType << "\"" << endl;
		cout << "IEnumEntry *enum_entry =  ptrPixelFormat->GetEntryByName(strNameFromPixelType);" << endl;
		IEnumEntry *enum_entry =  ptrPixelFormat->GetEntryByName(strNameFromPixelType);
		if (enum_entry == NULL)
		{
			cout << "    IEnumEntry not found." << endl << endl;
		}
		else
		{
			cout << "    IEnumEntry: \"" << enum_entry->GetNode()->GetName() << "\"" << endl << endl;

			cout << "Get GenICam PixelFormat Value from IEnumEntry::GetValue()." << endl;
			cout << "    int64_t int_value = enum_entry->GetValue();" << endl;
			int64_t int_value = enum_entry->GetValue();
			cout << "        Value: 0x" << hex << int_value << endl;
		}

        // Close the camera.
        camera.Close();
    }
    catch (const GenericException &e)
    {
        // Error handling.
        cerr << "An exception occurred." << endl
        << e.GetDescription() << endl;
        exitCode = 1;
    }

    // Comment the following two lines to disable waiting on exit.
    cerr << endl << "Press Enter to exit." << endl;
    while( cin.get() != '\n');

    // Releases all pylon resources. 
    PylonTerminate(); 

    return exitCode;
}

