// Grab.cpp
/*
    Note: Before getting started, Basler recommends reading the Programmer's Guide topic
    in the pylon C++ API documentation that gets installed with pylon.
    If you are upgrading to a higher major version of pylon, Basler also
    strongly recommends reading the Migration topic in the pylon C++ API documentation.

    This sample illustrates how to grab and process images using the CInstantCamera class.
    The images are grabbed and processed asynchronously, i.e.,
    while the application is processing a buffer, the acquisition of the next buffer is done
    in parallel.

    The CInstantCamera class uses a pool of buffers to retrieve image data
    from the camera device. Once a buffer is filled and ready,
    the buffer can be retrieved from the camera object for processing. The buffer
    and additional image data are collected in a grab result. The grab result is
    held by a smart pointer after retrieval. The buffer is automatically reused
    when explicitly released or when the smart pointer object is destroyed.
*/

// Include files to use the PYLON API.
#include <pylon/PylonIncludes.h>
#ifdef PYLON_WIN_BUILD
#    include <pylon/PylonGUI.h>
#endif

// Namespace for using pylon objects.
using namespace Pylon;

// Namespace for using cout.
using namespace std;

// Number of images to be grabbed.
static const uint32_t c_countOfImagesToGrab = 1;

int main(int argc, char* argv[])
{
    // The exit code of the sample application.
    int exitCode = 0;
	
	// Before using any pylon methods, the pylon runtime must be initialized. 
    PylonInitialize();
	CImageFormatConverter Converter;

    try
    {
        // Create an instant camera object with the camera device found first.
        CInstantCamera camera( CTlFactory::GetInstance().CreateFirstDevice());
		
        // Print the model name of the camera.
		cout << "Using device " << camera.GetDeviceInfo().GetModelName() << " // SN " << camera.GetDeviceInfo().GetSerialNumber() << endl << endl;

        // The parameter MaxNumBuffer can be used to control the count of buffers
        // allocated for grabbing. The default value of this parameter is 10.
        camera.MaxNumBuffer = 5;

        // Start the grabbing of c_countOfImagesToGrab images.
        // The camera device is parameterized with a default configuration which
        // sets up free-running continuous acquisition.
        camera.StartGrabbing( c_countOfImagesToGrab);
		
        // This smart pointer will receive the grab result data.
        CGrabResultPtr ptrGrabResult;

        // Camera.StopGrabbing() is called automatically by the RetrieveResult() method
        // when c_countOfImagesToGrab images have been retrieved.
        while ( camera.IsGrabbing())
        {
            // Wait for an image and then retrieve it. A timeout of 5000 ms is used.
            camera.RetrieveResult( 5000, ptrGrabResult, TimeoutHandling_ThrowException);

            // Image grabbed successfully?
            if (ptrGrabResult->GrabSucceeded())
            {

				CPylonImage targetImageBU; // BU stands for BottomUp
				CPylonImage targetImageTD; // TD stands for TopDown

                // Access the image data.
				cout << "SizeX: " << ptrGrabResult->GetWidth() << " // SizeY: " << ptrGrabResult->GetHeight() << endl;
                const uint8_t *pImageBuffer = (uint8_t *) ptrGrabResult->GetBuffer();
                cout << "Gray value of first pixel: " << (uint32_t) pImageBuffer[0] << endl << endl;
				
				// Default orientation is TopDown -- Convert and save one picture to the hard drive
				Converter.OutputOrientation.SetValue(OutputOrientation_TopDown);
				Converter.Convert(targetImageTD, ptrGrabResult);
				cout << "Saved picture with TopDown orientation" << endl;
				targetImageTD.Save(ImageFileFormat_Tiff,"picture_TopDown.tiff");
				
				// The orientation is actually set with AttachUserBuffer
				// We take the TopDown image and attach it as BottomUp
				targetImageBU.AttachUserBuffer(targetImageTD.GetBuffer(), ptrGrabResult->GetPayloadSize(), targetImageTD.GetPixelType(), ptrGrabResult->GetWidth(),
				ptrGrabResult->GetHeight(), ptrGrabResult->GetPaddingX(), ImageOrientation_BottomUp);
				cout << "Saved picture with BottomUp orientation" << endl;
				targetImageBU.Save(ImageFileFormat_Tiff,"picture_BottomUp.tiff");
				
				// Display both images
				Pylon::DisplayImage(1, targetImageTD);		
				Pylon::DisplayImage(2, targetImageBU);			
				

            }
            else
            {
                cout << "Error: " << ptrGrabResult->GetErrorCode() << " " << ptrGrabResult->GetErrorDescription() << endl;
            }
        }
    }
    catch (const GenericException &e)
    {
        // Error handling.
        cerr << "An exception occurred." << endl
        << e.GetDescription() << endl;
        exitCode = 1;
    }

    // Comment the following two lines to disable waiting on exit.
    cerr << endl << "Press Enter to exit." << endl;
    while( cin.get() != '\n');

    // Releases all pylon resources. 
    PylonTerminate();  

    return exitCode;
}
