﻿using System;
using System.Windows.Forms;
using Basler.Pylon;

namespace PylonLiveViewControl
{
    public partial class DualSliderUserControl : UserControl
    {
        private IFloatParameter parameterFloat = null; // The interface of the float parameter.
        private IIntegerParameter parameterInt = null; // The interface of the Int parameter.
        private string defaultName = "N/A";
        private bool sliderMoving = false;
        public DualSliderUserControl()
        {
            InitializeComponent();
        }
        #region floatParameter
        // Sets the parameter displayed by the user control.
        public IFloatParameter ParameterFloat
        {
            set
            {
                // Remove the old parameter.
                if (parameterFloat != null)
                {
                    parameterFloat.ParameterChanged -= ParameterChangedFloat;
                    slider.Scroll -= slider_Scroll;
                }

                // Set the new parameter.
                parameterFloat = value;
                if (parameterFloat != null)
                {
                    parameterFloat.ParameterChanged += ParameterChangedFloat;
                    slider.Scroll += slider_Scroll;
                    labelName.Text = parameterFloat.Advanced.GetPropertyOrDefault(AdvancedParameterAccessKey.DisplayName, parameterFloat.Name);
                    UpdateValuesFloat();
                }
                else
                {
                    labelName.Text = DefaultNameFloat;
                    Reset();
                }
            }
        }


        // Sets the default name of the control.
        public string DefaultNameFloat
        {
            set
            {
                defaultName = value;
                if (parameterFloat == null)
                {
                    labelName.Text = defaultName;
                }
            }
            get
            {
                return defaultName;
            }
        }


        // Occurs when the parameter state has changed. Updates the control.
        private void ParameterChangedFloat(Object sender, EventArgs e)
        {
            if (InvokeRequired)
            {
                // If called from a different thread, we must use the Invoke method to marshal the call to the proper thread.
                BeginInvoke(new EventHandler<EventArgs>(ParameterChangedFloat), sender, e);
                return;
            }
            try
            {
                // Update accessibility and parameter values.
                UpdateValuesFloat();
            }
            catch
            {
                // If errors occurred disable the control.
                Reset();
            }
        }

        // Deactivate the control and deregister the callback.
        private void Reset()
        {
            slider.Enabled = false;
            labelMin.Enabled = false;
            labelMax.Enabled = false;
            labelName.Enabled = false;
            labelCurrentValue.Enabled = false;
        }


        // Converts slider range to percent value.
        private int PercentToSliderValue(double percent)
        {
            return (int)((100000.0 / 100.0) * percent);
        }


        // Converts percent value to slider range.
        private double SliderToPercentValue(int sliderValue)
        {
            return (((double)sliderValue) / 100000.0) * 100.0;
        }


        // Gets the current values from the node and displays them.
        private void UpdateValuesFloat()
        {
            try
            {
                if (parameterFloat != null)
                {
                    if (parameterFloat.IsReadable)  // Check if the parameterFloat is readable.
                    {
                        // Get the values.
                        double min = parameterFloat.GetMinimum();
                        double max = parameterFloat.GetMaximum();
                        double val = parameterFloat.GetValue();
                        double percent = parameterFloat.GetValuePercentOfRange();

                        // Update the slider.
                        slider.Minimum = PercentToSliderValue(0);
                        slider.Maximum = PercentToSliderValue(100);
                        slider.Value = PercentToSliderValue(percent);
                        slider.SmallChange = PercentToSliderValue(0.05);
                        slider.TickFrequency = PercentToSliderValue(10);

                        // Update the displayed values.
                        labelMin.Text = "" + min;
                        labelMax.Text = "" + max;
                        labelCurrentValue.Text = "" + val;

                        // Update the access status.
                        slider.Enabled = parameterFloat.IsWritable;
                        labelMin.Enabled = true;
                        labelMax.Enabled = true;
                        labelName.Enabled = true;
                        labelCurrentValue.Enabled = true;

                        return;
                    }
                }
            }
            catch
            {
                // If errors occurred, disable the control.
            }
            Reset();

        }
        #endregion  

        #region IntParameter
        // Sets the parameter displayed by the user control.
        public IIntegerParameter ParameterInt
        {
            set
            {
                // Remove the old parameterInt.
                if (parameterInt != null)
                {
                    parameterInt.ParameterChanged -= ParameterChangedInt;
                    slider.Scroll -= slider_Scroll;
                }

                // Set the new parameterInt.
                parameterInt = value;
                if (parameterInt != null)
                {
                    parameterInt.ParameterChanged += ParameterChangedInt;
                    slider.Scroll += slider_Scroll;
                    labelName.Text = parameterInt.Advanced.GetPropertyOrDefault(AdvancedParameterAccessKey.DisplayName, parameterInt.Name);
                    UpdateValuesInt();
                }
                else
                {
                    labelName.Text = defaultName;
                    Reset();
                }
            }
        }


        // Sets the default name of the control.
        public string DefaultNameInt
        {
            set
            {
                defaultName = value;
                if (parameterInt == null)
                {
                    labelName.Text = defaultName;
                }
            }
            get
            {
                return defaultName;
            }
        }


        // The parameter state changed. Update the control.
        private void ParameterChangedInt(Object sender, EventArgs e)
        {
            if (InvokeRequired)
            {
                // If called from a different thread, we must use the Invoke method to marshal the call to the proper thread.
                BeginInvoke(new EventHandler<EventArgs>(ParameterChangedInt), sender, e);
                return;
            }
            try
            {
                UpdateValuesInt();
            }
            catch
            {
                // If errors occurred disable the control.
                Reset();
            }
        }


        // Get the current values from the parameter and display them.
        private void UpdateValuesInt()
        {
            try
            {
                if (parameterInt != null)
                {
                    if (parameterInt.IsReadable)  // Check if parameter is accessible.
                    {
                        // Get values.
                        int min = checked((int)parameterInt.GetMinimum());
                        int max = checked((int)parameterInt.GetMaximum());
                        int val = checked((int)parameterInt.GetValue());
                        int inc = checked((int)parameterInt.GetIncrement());

                        // Update the slider.
                        slider.Minimum = min;
                        slider.Maximum = max;
                        slider.Value = val;
                        slider.SmallChange = inc;
                        slider.TickFrequency =  (max - min + 5) / 10;

                        // Update the displayed values.
                        labelMin.Text = "" + min;
                        labelMax.Text = "" + max;
                        labelCurrentValue.Text = "" + val;

                        // Update accessibility.
                        slider.Enabled = parameterInt.IsWritable;
                        labelMin.Enabled = true;
                        labelMax.Enabled = true;
                        labelName.Enabled = true;
                        labelCurrentValue.Enabled = true;

                        return;
                    }
                }
            }
            catch
            {
                // If errors occurred disable the control.
            }
            Reset();
        }

        #endregion

        // Occurs when the slider position changes.
        private void slider_Scroll(object sender, EventArgs e)
        {
            if (parameterFloat != null)
            {
                try
                {
                    if (parameterFloat.IsWritable && !sliderMoving)
                    {
                        // Break any recursion if the value does not exactly match the slider value.
                        sliderMoving = true;

                        // Set the value.
                        int x = (int)SliderToPercentValue(slider.Value);
                        parameterFloat.SetValuePercentOfRange(SliderToPercentValue(slider.Value));
                    }
                }
                catch
                {
                    // Ignore any errors here.
                }
                finally
                {
                    sliderMoving = false;
                }


            }
            if (parameterInt != null)
            {
                try
                {
                    if (parameterInt.IsWritable && !sliderMoving)
                    {
                        // Break any recursion if the value does not exactly match the slider value.
                        sliderMoving = true;

                        // Set the value.
                        int x = (int)SliderToPercentValue(slider.Value);
                        //parameterInt.SetValuePercentOfRange(SliderToPercentValue(slider.Value));
                        parameterInt.TrySetValue(slider.Value);
                    }
                }
                catch
                {
                    // Ignore any errors here.
                }
                finally
                {
                    sliderMoving = false;
                }
            }

        }
    }
}
