/*
   This sample illustrates how to grab and process images asynchronously, i.e.,
   while the application is processing a buffer, the acquistion of the next buffer is done
   in parallel.
   The sample uses a pool of buffers that are passed to a stream grabber to be filled with
   image data. Once a buffer is filled and ready for processing, the buffer is retrieved from
   the stream grabber, processed, and passed back to the stream grabber to be filled again.
   Buffers retrieved from the stream grabber are not overwritten as long as
   they are not passed back to the stream grabber.

   This sample also illustrates, how to read the stream grabber statistics.
*/



#include <stdlib.h>
#include <stdio.h>
#include <malloc.h>

#include <pylonc/PylonC.h>

#define CHECK( errc ) if ( GENAPI_E_OK != errc ) printErrorAndExit( errc )

/* This function can be used to wait for user input at the end of the sample program. */
void pressEnterToExit( void );

/* This method demonstrates how to retrieve the error message for the last failed function call. */
void printErrorAndExit( GENAPIC_RESULT errc );

/* Calculating the minimum and maximum gray value */
void getMinMax( const unsigned char* pImg, int32_t width, int32_t height,
                unsigned char* pMin, unsigned char* pMax );



//
//
//
GENAPIC_RESULT IterateThroughCategory(NODE_HANDLE hCategory);
//
//
//



#define NUM_GRABS 100         /* Number of images to grab. */
#define NUM_BUFFERS 5         /* Number of buffers used for grabbing. */

int main( void )
{
    GENAPIC_RESULT              res;                      /* Return value of pylon methods. */
    size_t                      numDevices;               /* Number of available devices. */
    PYLON_DEVICE_HANDLE         hDev;                     /* Handle for the pylon device. */
 
    
    PYLON_STREAMGRABBER_HANDLE  hStreamGrabber;           /* Handle for the pylon stream grabber. */

    //
    //
    //
    /* Stream grabber nodemap access (for stream grabber statistics) */
    NODEMAP_HANDLE              hStreamGrabberNodeMap;      /* Handle for the stream grabber nodemap */


    /* Stream grabber Statistics */

    /* All cameras */
    NODE_HANDLE                 hStatisticTotalBufferCount;
    NODE_HANDLE                 hStatisticFailedBufferCount;

    /* USB cameras */
    NODE_HANDLE                 hStatisticLastFailedBufferStatus;
    NODE_HANDLE                 hStatisticLastFailedBufferStatusText;
    NODE_HANDLE                 hStatisticMissedFrameCount;
    NODE_HANDLE                 hStatisticResynchronizationCount;
    NODE_HANDLE                 hStatisticLastBlockId;
    NODE_HANDLE                 hStatisticOutOfMemoryErrorCount;

    /* GigE cameras */
    NODE_HANDLE                 hStatisticBufferUnderrunCount;
    NODE_HANDLE                 hStatisticTotalPacketCount;
    NODE_HANDLE                 hStatisticFailedPacketCount;
    NODE_HANDLE                 hStatisticResendRequestCount;
    NODE_HANDLE                 hStatisticResendPacketCount;
    //
    //
    //

    PYLON_WAITOBJECT_HANDLE     hWait;                    /* Handle used for waiting for a grab to be finished. */
    size_t                      payloadSize;              /* Size of an image frame in bytes. */
    unsigned char*              buffers[NUM_BUFFERS];     /* Buffers used for grabbing. */
    PYLON_STREAMBUFFER_HANDLE   bufHandles[NUM_BUFFERS];  /* Handles for the buffers. */
    PylonGrabResult_t           grabResult;               /* Stores the result of a grab operation. */
    int                         nGrabs;                   /* Counts the number of buffers grabbed. */
    size_t                      nStreams;                 /* The number of streams the device provides. */
    _Bool                       isAvail;                  /* Used for checking feature availability. */
    _Bool                       isReady;                  /* Used as an output parameter. */
    size_t                      i;                        /* Counter. */

    char string_buf[256];
    size_t string_buf_size = sizeof(string_buf);
    _Bool isReadable;

    /* Before using any pylon methods, the pylon runtime must be initialized. */
    PylonInitialize();

    /* Enumerate all camera devices. You must call
    PylonEnumerateDevices() before creating a device. */
    res = PylonEnumerateDevices( &numDevices );
    CHECK( res );
    if (0 == numDevices)
    {
        fprintf( stderr, "No devices found.\n" );
        PylonTerminate();
        pressEnterToExit();
        exit( EXIT_FAILURE );
    }

    /* Get a handle for the first device found.  */
    res = PylonCreateDeviceByIndex( 0, &hDev );
    CHECK( res );

    /* Before using the device, it must be opened. Open it for configuring
    parameters and for grabbing images. */
    res = PylonDeviceOpen( hDev, PYLONC_ACCESS_MODE_CONTROL | PYLONC_ACCESS_MODE_STREAM );
    CHECK( res );

    /* Print out the name of the camera we are using. */
    isReadable = PylonDeviceFeatureIsReadable( hDev, "DeviceModelName" );
    if (isReadable)
    {
        res = PylonDeviceFeatureToString( hDev, "DeviceModelName", string_buf, &string_buf_size );
        CHECK( res );
        printf( "Using camera %s\n", string_buf );
    }

    /* Set the pixel format to Mono8 if available, where gray values will be output as 8 bit values for each pixel. */
    isAvail = PylonDeviceFeatureIsAvailable( hDev, "EnumEntry_PixelFormat_Mono8" );
    if (isAvail)
    {
        res = PylonDeviceFeatureFromString( hDev, "PixelFormat", "Mono8" );
        CHECK( res );
    }

    /* Disable acquisition start trigger if available. */
    isAvail = PylonDeviceFeatureIsAvailable( hDev, "EnumEntry_TriggerSelector_AcquisitionStart" );
    if (isAvail)
    {
        res = PylonDeviceFeatureFromString( hDev, "TriggerSelector", "AcquisitionStart" );
        CHECK( res );
        res = PylonDeviceFeatureFromString( hDev, "TriggerMode", "Off" );
        CHECK( res );
    }

    /* Disable frame burst start trigger if available. */
    isAvail = PylonDeviceFeatureIsAvailable( hDev, "EnumEntry_TriggerSelector_FrameBurstStart" );
    if (isAvail)
    {
        res = PylonDeviceFeatureFromString( hDev, "TriggerSelector", "FrameBurstStart" );
        CHECK( res );
        res = PylonDeviceFeatureFromString( hDev, "TriggerMode", "Off" );
        CHECK( res );
    }

    /* Disable frame start trigger if available. */
    isAvail = PylonDeviceFeatureIsAvailable( hDev, "EnumEntry_TriggerSelector_FrameStart" );
    if (isAvail)
    {
        res = PylonDeviceFeatureFromString( hDev, "TriggerSelector", "FrameStart" );
        CHECK( res );
        res = PylonDeviceFeatureFromString( hDev, "TriggerMode", "Off" );
        CHECK( res );
    }

    /* We will use the Continuous frame acquisition mode, i.e., the camera delivers
    images continuously. */
    res = PylonDeviceFeatureFromString( hDev, "AcquisitionMode", "Continuous" );
    CHECK( res );


    /* For GigE cameras, we recommend increasing the packet size for better
       performance. When the network adapter supports jumbo frames, set the packet
       size to a value > 1500, e.g., to 8192. In this sample, we only set the packet size
       to 1500. */
    /* ... Check first to see if the GigE camera packet size parameter is supported and if it is writable. */
    isAvail = PylonDeviceFeatureIsWritable( hDev, "GevSCPSPacketSize" );
    if (isAvail)
    {
        /* ... The device supports the packet size feature, set a value. */
        res = PylonDeviceSetIntegerFeature( hDev, "GevSCPSPacketSize", 1500 );
        CHECK( res );
    }


    /* Image grabbing is done using a stream grabber.
      A device may be able to provide different streams. A separate stream grabber must
      be used for each stream. In this sample, we create a stream grabber for the default
      stream, i.e., the first stream ( index == 0 ).
      */

    /* Get the number of streams supported by the device and the transport layer. */
    res = PylonDeviceGetNumStreamGrabberChannels( hDev, &nStreams );
    CHECK( res );
    if (nStreams < 1)
    {
        fprintf( stderr, "The transport layer doesn't support image streams\n" );
        PylonTerminate();
        pressEnterToExit();
        exit( EXIT_FAILURE );
    }

    /* Create and open a stream grabber for the first channel. */
    res = PylonDeviceGetStreamGrabber( hDev, 0, &hStreamGrabber );
    CHECK( res );
    res = PylonStreamGrabberOpen( hStreamGrabber );
    CHECK( res );

    /* Get a handle for the stream grabber's wait object. The wait object
       allows waiting for buffers to be filled with grabbed data. */
    res = PylonStreamGrabberGetWaitObject( hStreamGrabber, &hWait );
    CHECK( res );


    /* Determine the minimum size of the grab buffer.
       The size is determined by the configuration of the camera
       and the stream grabber. Be aware that this may change
       by changing critical parameters after this call.*/
    res = PylonStreamGrabberGetPayloadSize( hDev, hStreamGrabber, &payloadSize );
    CHECK( res );

    /* Allocate memory for grabbing.  */
    for (i = 0; i < NUM_BUFFERS; ++i)
    {
        buffers[i] = (unsigned char*) malloc( payloadSize );
        if (NULL == buffers[i])
        {
            fprintf( stderr, "Out of memory!\n" );
            PylonTerminate();
            pressEnterToExit();
            exit( EXIT_FAILURE );
        }
    }

    /* We must tell the stream grabber the number and size of the buffers
        we are using. */
    /* .. We will not use more than NUM_BUFFERS for grabbing. */
    res = PylonStreamGrabberSetMaxNumBuffer( hStreamGrabber, NUM_BUFFERS );
    CHECK( res );
    /* .. We will not use buffers bigger than payloadSize bytes. */
    res = PylonStreamGrabberSetMaxBufferSize( hStreamGrabber, payloadSize );
    CHECK( res );


    /*  Allocate the resources required for grabbing. After this, critical parameters
        that impact the payload size must not be changed until FinishGrab() is called. */
    res = PylonStreamGrabberPrepareGrab( hStreamGrabber );
    CHECK( res );


    /* Before using the buffers for grabbing, they must be registered at
       the stream grabber. For each registered buffer, a buffer handle
       is returned. After registering, these handles are used instead of the
       raw pointers. */
    for (i = 0; i < NUM_BUFFERS; ++i)
    {
        res = PylonStreamGrabberRegisterBuffer( hStreamGrabber, buffers[i], payloadSize, &bufHandles[i] );
        CHECK( res );
    }

    /* Feed the buffers into the stream grabber's input queue. For each buffer, the API
       allows passing in a pointer to additional context information. This pointer
       will be returned unchanged when the grab is finished. In our example, we use the index of the
       buffer as context information. */
    for (i = 0; i < NUM_BUFFERS; ++i)
    {
        res = PylonStreamGrabberQueueBuffer( hStreamGrabber, bufHandles[i], (void*) i );
        CHECK( res );
    }

    /* Now the stream grabber is prepared. As soon as the camera starts to acquire images,
       the image data will be grabbed into the buffers provided.  */

    /* Start the image acquisition engine. */
    res = PylonStreamGrabberStartStreamingIfMandatory( hStreamGrabber );
    CHECK( res );

    /* Let the camera acquire images. */
    res = PylonDeviceExecuteCommandFeature( hDev, "AcquisitionStart" );
    CHECK( res );

    /* Grab NUM_GRABS images */
    nGrabs = 0;                         /* Counts the number of images grabbed */


    //
    //
    //
    /* Get stream grabber nodemap for stream grabber statistics */
    res = PylonStreamGrabberGetNodeMap(hStreamGrabber, &hStreamGrabberNodeMap);
    CHECK( res );

    /* Get stream grabber Statistic nodes */
    res = GenApiNodeMapGetNode( hStreamGrabberNodeMap, "Statistic_Total_Buffer_Count", &hStatisticTotalBufferCount );
    CHECK( res );
    res = GenApiNodeMapGetNode( hStreamGrabberNodeMap, "Statistic_Failed_Buffer_Count", &hStatisticFailedBufferCount );
    CHECK( res );
    /* USB cameras */
    res = GenApiNodeMapGetNode( hStreamGrabberNodeMap, "Statistic_Last_Failed_Buffer_Status", &hStatisticLastFailedBufferStatus );
    CHECK( res );
    res = GenApiNodeMapGetNode( hStreamGrabberNodeMap, "Statistic_Last_Failed_Buffer_Status_Text", &hStatisticLastFailedBufferStatusText );
    CHECK( res );
    res = GenApiNodeMapGetNode( hStreamGrabberNodeMap, "Statistic_Missed_Frame_Count", &hStatisticMissedFrameCount );
    CHECK( res );
    res = GenApiNodeMapGetNode( hStreamGrabberNodeMap, "Statistic_Resynchronization_Count", &hStatisticResynchronizationCount );
    CHECK( res );
    res = GenApiNodeMapGetNode( hStreamGrabberNodeMap, "Statistic_Last_Block_Id", &hStatisticLastBlockId );
    CHECK( res );
    res = GenApiNodeMapGetNode( hStreamGrabberNodeMap, "Statistic_Out_Of_Memory_Error_Count", &hStatisticOutOfMemoryErrorCount);
    CHECK( res );
    /* GigE cameras */
    res = GenApiNodeMapGetNode(hStreamGrabberNodeMap, "Statistic_Buffer_Underrun_Count", &hStatisticBufferUnderrunCount);
    CHECK(res);
    res = GenApiNodeMapGetNode(hStreamGrabberNodeMap, "Statistic_Total_Packet_Count", &hStatisticTotalPacketCount);
    CHECK(res);
    res = GenApiNodeMapGetNode(hStreamGrabberNodeMap, "Statistic_Failed_Packet_Count", &hStatisticFailedPacketCount);
    CHECK(res);
    res = GenApiNodeMapGetNode(hStreamGrabberNodeMap, "Statistic_Resend_Request_Count", &hStatisticResendRequestCount);
    CHECK(res);
    res = GenApiNodeMapGetNode(hStreamGrabberNodeMap, "Statistic_Resend_Packet_Count", &hStatisticResendPacketCount);
    CHECK(res);
    //
    //
    //

    while (nGrabs < NUM_GRABS)
    {
        size_t bufferIndex;              /* Index of the buffer */
        unsigned char min, max;
        /* Wait for the next buffer to be filled. Wait up to 1000 ms. */
        res = PylonWaitObjectWait( hWait, 1000, &isReady );
        CHECK( res );
        if (!isReady)
        {
            /* Timeout occurred. */
            fprintf( stderr, "Grab timeout occurred\n" );
            break; /* Stop grabbing. */
        }

        /* Since the wait operation was successful, the result of at least one grab
           operation is available. Retrieve it. */
        res = PylonStreamGrabberRetrieveResult( hStreamGrabber, &grabResult, &isReady );
        CHECK( res );
        if (!isReady)
        {
            /* Oops. No grab result available? We should never have reached this point.
               Since the wait operation above returned without a timeout, a grab result
               should be available. */
            fprintf( stderr, "Failed to retrieve a grab result\n" );
            break;
        }

        nGrabs++;

        /* Get the buffer index from the context information. */
        bufferIndex = (size_t) grabResult.Context;

        /* Check to see if the image was grabbed successfully. */
        if (grabResult.Status == Grabbed)
        {
            /*  Success. Perform image processing. Since we passed more than one buffer
            to the stream grabber, the remaining buffers are filled while
            we do the image processing. The processed buffer won't be touched by
            the stream grabber until we pass it back to the stream grabber. */

            unsigned char* buffer;        /* Pointer to the buffer attached to the grab result. */

            /* Get the buffer pointer from the result structure. Since we also got the buffer index,
               we could alternatively use buffers[bufferIndex]. */
            buffer = (unsigned char*) grabResult.pBuffer;

            /* Perform processing. */
            getMinMax( buffer, grabResult.SizeX, grabResult.SizeY, &min, &max );
            printf( "Grabbed frame %2d into buffer %2d. Min. gray value = %3u, Max. gray value = %3u\n",
                    nGrabs, (int) bufferIndex, min, max );

#ifdef GENAPIC_WIN_BUILD
            /* Display image */
            res = PylonImageWindowDisplayImageGrabResult( 0, &grabResult );
            CHECK( res );
#endif

        }
        else if (grabResult.Status == Failed)
        {
            fprintf( stderr, "Frame %d wasn't grabbed successfully.  Error code = 0x%08X\n",
                     nGrabs, grabResult.ErrorCode );
        }

        //
        //
        //
        /* Print out some stream grabber statistics */
        res = GenApiNodeIsReadable(hStatisticTotalBufferCount, &isReadable);
        CHECK(res);
        if (isReadable)
        {
            /* Read node value as integer */
            int64_t int_value = 0;
            res = GenApiIntegerGetValue(hStatisticTotalBufferCount, &int_value);
            CHECK(res);

            /* Read node value as string */
            string_buf_size = sizeof(string_buf);
            res = GenApiNodeToString(hStatisticTotalBufferCount, string_buf, &string_buf_size);
            CHECK(res);
        }
        /* Continue with other Statistic nodes */

        /* Or iterate through Category "Statistic" and print out all node values */
        NODE_HANDLE hCategoryStatistic;
        res = GenApiNodeMapGetNode(hStreamGrabberNodeMap, "Statistic", &hCategoryStatistic);
        CHECK(res);
        res = IterateThroughCategory(hCategoryStatistic);
        CHECK(res);
        //
        //
        //

        /* Once finished with the processing, requeue the buffer to be filled again. */
        res = PylonStreamGrabberQueueBuffer( hStreamGrabber, grabResult.hBuffer, (void*) bufferIndex );
        CHECK( res );
    }

    /* Clean up. */

    /*  ... Stop the camera. */
    res = PylonDeviceExecuteCommandFeature( hDev, "AcquisitionStop" );
    CHECK( res );

    /* ... Stop the image acquisition engine. */
    res = PylonStreamGrabberStopStreamingIfMandatory( hStreamGrabber );
    CHECK( res );

    /* ... We must issue a flush call to ensure that all pending buffers are put into the
       stream grabber's output queue. */
    res = PylonStreamGrabberFlushBuffersToOutput( hStreamGrabber );
    CHECK( res );

    /* ... The buffers can now be retrieved from the stream grabber. */
    do
    {
        res = PylonStreamGrabberRetrieveResult( hStreamGrabber, &grabResult, &isReady );
        CHECK( res );
    } while (isReady);

    /* ... When all buffers have been retrieved from the stream grabber, they can be deregistered.
           After that, it is safe to free the memory. */

    for (i = 0; i < NUM_BUFFERS; ++i)
    {
        res = PylonStreamGrabberDeregisterBuffer( hStreamGrabber, bufHandles[i] );
        CHECK( res );
        free( buffers[i] );
    }

    /* ... Release grabbing related resources. */
    res = PylonStreamGrabberFinishGrab( hStreamGrabber );
    CHECK( res );

    /* After calling PylonStreamGrabberFinishGrab(), parameters that impact the payload size (e.g.,
    the AOI width and height parameters) are unlocked and can be modified again. */

    /* ... Close the stream grabber. */
    res = PylonStreamGrabberClose( hStreamGrabber );
    CHECK( res );


    /* ... Close and release the pylon device. The stream grabber becomes invalid
       after closing the pylon device. Don't call stream grabber related methods after
       closing or releasing the device. */
    res = PylonDeviceClose( hDev );
    CHECK( res );

    /* ...The device is no longer used, destroy it. */
    res = PylonDestroyDevice( hDev );
    CHECK( res );

    pressEnterToExit();

    /* ... Shut down the pylon runtime system. Don't call any pylon method after
       calling PylonTerminate(). */
    PylonTerminate();


    return EXIT_SUCCESS;
}


/* This iterates through a given Category and prints all feature names (except (Sub-)Categories) and their values (as string) */
GENAPIC_RESULT IterateThroughCategory(NODE_HANDLE hCategory)
{
    _Bool isReadable = FALSE;
    _Bool isAvailable = FALSE;

    char string_buf[256];
    size_t string_buf_size = sizeof(string_buf);

    GENAPIC_RESULT res;

    res = GenApiNodeIsAvailable(hCategory, &isAvailable);
    CHECK(res);
    if (isAvailable)
    {
        res = GenApiNodeIsReadable(hCategory, &isReadable);
        CHECK(res);
        if (isReadable)
        {
            string_buf_size = sizeof(string_buf);
            res = GenApiNodeGetDisplayName(hCategory, string_buf, &string_buf_size);
            CHECK(res);

            EGenApiNodeType node_type;
            res = GenApiNodeGetType(hCategory, &node_type);
            CHECK(res);

            if (Category != node_type)
            {
                printf("IterateThroughCategory(): Node \"%s\" is not of type \"Category\"!", string_buf);
                res = GENAPI_E_INVALID_NODEHANDLE;
            }
            else
            {

                printf("\"%s\":\n", string_buf);

                int64_t num_features;
                res = GenApiCategoryGetNumFeatures(hCategory, &num_features);
                for (int i = 0; i < num_features; i++)
                {
                    NODE_HANDLE hNode;
                    res = GenApiCategoryGetFeatureByIndex(hCategory, i, &hNode);
                    CHECK(res);
                    res = GenApiNodeGetType(hNode, &node_type);
                    CHECK(res);
                    
                    /* Skip (Sub-)Categories, only print features */
                    if (Category != node_type)
                    {
                        res = GenApiNodeIsReadable(hNode, &isReadable);
                        CHECK(res);
                        if (isReadable)
                        {
                            /* Print the nodes display name */
                            string_buf_size = sizeof(string_buf);
                            res = GenApiNodeGetDisplayName(hNode, string_buf, &string_buf_size);
                            CHECK(res);
                            printf("\t\"%s\": ", string_buf);
                            
                            /* Print the nodes value */
                            string_buf_size = sizeof(string_buf);
                            res = GenApiNodeToString(hNode, string_buf, &string_buf_size);
                            CHECK(res);
                            printf("%s\n", string_buf);
                        }
                    }
                }
                printf("\n");
            }
        }
    }
    return res;
}





/* This function demonstrates how to retrieve the error message for the last failed
   function call. */
void printErrorAndExit( GENAPIC_RESULT errc )
{
    char* errMsg;
    size_t length;

    /* Retrieve the error message.
    ... First find out how big the buffer must be, */
    GenApiGetLastErrorMessage( NULL, &length );
    errMsg = (char*) malloc( length );
    /* ... and retrieve the message. */
    GenApiGetLastErrorMessage( errMsg, &length );

    fprintf( stderr, "%s (%#08x).\n", errMsg, (unsigned int) errc );
    free( errMsg );

    /* Retrieve more details about the error.
    ... First find out how big the buffer must be, */
    GenApiGetLastErrorDetail( NULL, &length );
    errMsg = (char*) malloc( length );
    /* ... and retrieve the message. */
    GenApiGetLastErrorDetail( errMsg, &length );

    fprintf( stderr, "%s\n", errMsg );
    free( errMsg );

    PylonTerminate();  /* Releases all pylon resources */
    pressEnterToExit();

    exit( EXIT_FAILURE );
}



/* Simple "image processing" function returning the minimum and maximum gray
   value of an 8 bit gray value image. */
void getMinMax( const unsigned char* pImg, int32_t width, int32_t height,
                unsigned char* pMin, unsigned char* pMax )
{
    unsigned char min = 255;
    unsigned char max = 0;
    unsigned char val;
    const unsigned char* p;

    for (p = pImg; p < pImg + width * height; p++)
    {
        val = *p;
        if (val > max)
            max = val;
        if (val < min)
            min = val;
    }
    *pMin = min;
    *pMax = max;
}

/* This function can be used to wait for user input at the end of the sample program. */
void pressEnterToExit( void )
{
    fprintf( stderr, "\nPress enter to exit.\n" );
    while (getchar() != '\n');
}

