/*
   This sample illustrates how to grab images and process images asynchronously.
   This means that while the application is processing a buffer,
   the acquisition of the next buffer is done in parallel.
   The sample uses a pool of buffers. The buffers are automatically allocated. Once a buffer is filled
   and ready for processing, the buffer is retrieved from the stream grabber as part of a grab
   result. The grab result is processed and the buffer is passed back to the stream grabber by
   disposing the grab result. The buffer is reused and refilled.
   A buffer retrieved from the stream grabber as a grab result is not overwritten in the background
   as long as the grab result is not disposed.
*/

using System;
using Basler.Pylon;

namespace Grab_ThreadPriority
{
    class Grab_ThreadPriority
    {
        internal static void Main()
        {
            // The exit code of the sample application.
            int exitCode = 0;

            try
            {
				// Create a camera object that selects the first camera device found.
				// More constructors are available for selecting a specific camera device.
				using (Camera camera = new Camera())
				{
					// Print the model name of the camera.
					Console.WriteLine("Using camera {0}.", camera.CameraInfo[CameraInfoKey.ModelName]);

					// Set the acquisition mode to free running continuous acquisition when the camera is opened.
					camera.CameraOpened += Configuration.AcquireContinuous;

					// Open the connection to the camera device.
					camera.Open();
					string device_type = camera.CameraInfo[CameraInfoKey.DeviceType];
					Console.WriteLine("CameraInfoKey.DeviceType is: \"{0}\"", device_type);
					if (device_type == "BaslerGigE")
					{
						string gige_type = camera.Parameters[PLGigEStream.Type].GetValueOrDefault("<Parameter not available>");
						Console.WriteLine("PLGigEStream.Type is: \"{0}\"", gige_type);
						long gige_stream_type_is_socket_driver_available = camera.Parameters[PLGigEStream.TypeIsSocketDriverAvailable].GetValueOrDefault(-999);
						Console.WriteLine("PLGigEStream.TypeIsSocketDriverAvailable is: \"{0}\"", gige_stream_type_is_socket_driver_available);
						if (gige_stream_type_is_socket_driver_available == 1)
						{
							Console.WriteLine("Setting PLGigEStream.Type to: SocketDriver...");
							if (camera.Parameters[PLGigEStream.Type].TrySetValue("SocketDriver"))
							{
								gige_type = camera.Parameters[PLGigEStream.Type].GetValueOrDefault("<Parameter not available>");
								Console.WriteLine("PLGigEStream.Type now is: \"{0}\"", gige_type);
								Console.WriteLine();

								bool gige_receive_thread_priority_override_is_readable = camera.Parameters[PLGigEStream.ReceiveThreadPriorityOverride].IsReadable;
								bool gige_receive_thread_priority_override_is_writable = camera.Parameters[PLGigEStream.ReceiveThreadPriorityOverride].IsWritable;
								Console.WriteLine("PLGigEStream.ReceiveThreadPriorityOverride is: {0}.", ((gige_receive_thread_priority_override_is_readable == true) ? "READABLE" : "NOT READABLE"));
								Console.WriteLine("PLGigEStream.ReceiveThreadPriorityOverride is: {0}.", ((gige_receive_thread_priority_override_is_writable == true) ? "WRITABLE" : "NOT WRITABLE"));
								if (gige_receive_thread_priority_override_is_readable)
								{
									bool gige_receive_thread_priority_override = camera.Parameters[PLGigEStream.ReceiveThreadPriorityOverride].GetValue();
									Console.WriteLine("PLGigEStream.ReceiveThreadPriorityOverride is: \"{0}\"", gige_receive_thread_priority_override);
								}
								if (gige_receive_thread_priority_override_is_writable)
								{
									Console.WriteLine("Setting PLGigEStream.ReceiveThreadPriorityOverride to: True...");
									if (camera.Parameters[PLGigEStream.ReceiveThreadPriorityOverride].TrySetValue(true))
									{
										bool gige_receive_thread_priority_override = camera.Parameters[PLGigEStream.ReceiveThreadPriorityOverride].GetValue();
										Console.WriteLine("PLGigEStream.ReceiveThreadPriorityOverride now is: \"{0}\"", gige_receive_thread_priority_override);

										bool gige_receive_thread_priority_is_readable = camera.Parameters[PLGigEStream.ReceiveThreadPriority].IsReadable;
										bool gige_receive_thread_priority_is_writable = camera.Parameters[PLGigEStream.ReceiveThreadPriority].IsWritable;
										Console.WriteLine("PLGigEStream.ReceiveThreadPriority is: {0}.", ((gige_receive_thread_priority_is_readable == true) ? "READABLE" : "NOT READABLE"));
										Console.WriteLine("PLGigEStream.ReceiveThreadPriority is: {0}.", ((gige_receive_thread_priority_is_writable == true) ? "WRITABLE" : "NOT WRITABLE"));
										if (gige_receive_thread_priority_is_readable)
										{
											long gige_receive_thread_priority = camera.Parameters[PLGigEStream.ReceiveThreadPriority].GetValueOrDefault(-999);
											Console.WriteLine("PLGigEStream.ReceiveThreadPriority is: \"{0}\"", gige_receive_thread_priority);
											long min_priority = camera.Parameters[PLGigEStream.ReceiveThreadPriority].GetMinimum();
											long max_priority = camera.Parameters[PLGigEStream.ReceiveThreadPriority].GetMaximum();
										}
										if (gige_receive_thread_priority_is_writable)
										{
											long min_priority = camera.Parameters[PLGigEStream.ReceiveThreadPriority].GetMinimum();
											long max_priority = camera.Parameters[PLGigEStream.ReceiveThreadPriority].GetMaximum();
											Console.WriteLine("PLGigEStream.ReceiveThreadPriority Minimum is: {0}", min_priority);
											Console.WriteLine("PLGigEStream.ReceiveThreadPriority Maximum is: {0}", max_priority);
											Console.WriteLine("Setting PLGigEStream.ReceiveThreadPriority to: Maximum...");
											if (camera.Parameters[PLGigEStream.ReceiveThreadPriority].TrySetValue(max_priority))
											{
												long gige_receive_thread_priority = camera.Parameters[PLGigEStream.ReceiveThreadPriority].GetValueOrDefault(-999);
												Console.WriteLine("PLGigEStream.ReceiveThreadPriority now is: \"{0}\"", gige_receive_thread_priority);
											}
											else
											{
												Console.WriteLine("Failed to set PLGigEStream.ReceiveThreadPriority to: Maximum.");
											}
										}
										else
										{
											Console.WriteLine("Failed to set PLGigEStream.ReceiveThreadPriority to: True.");
										}
									}
								}
								else
								{
									Console.WriteLine("Failed to set PLGigEStream.Type to: SocketDriver.");
								}
							}

						}

						Console.WriteLine();

						// The parameter MaxNumBuffer can be used to control the amount of buffers
						// allocated for grabbing. The default value of this parameter is 10.
						camera.Parameters[PLCameraInstance.MaxNumBuffer].SetValue(5);

						// Start grabbing.
						camera.StreamGrabber.Start();

						// Grab a number of images.
						for (int i = 0; i < 10; ++i)
						{
							// Wait for an image and then retrieve it. A timeout of 5000 ms is used.
							IGrabResult grabResult = camera.StreamGrabber.RetrieveResult(5000, TimeoutHandling.ThrowException);
							using (grabResult)
							{
								// Image grabbed successfully?
								if (grabResult.GrabSucceeded)
								{
									// Access the image data.
									Console.WriteLine("SizeX: {0}", grabResult.Width);
									Console.WriteLine("SizeY: {0}", grabResult.Height);
									byte[] buffer = grabResult.PixelData as byte[];
									Console.WriteLine("Gray value of first pixel: {0}", buffer[0]);
									Console.WriteLine("");

									// Display the grabbed image.
									ImageWindow.DisplayImage(0, grabResult);
								}
								else
								{
									Console.WriteLine("Error: {0} {1}", grabResult.ErrorCode, grabResult.ErrorDescription);
								}
							}
						}

						// Stop grabbing.
						camera.StreamGrabber.Stop();

						// Close the connection to the camera device.
						camera.Close();
					}
				}
            }
            catch (Exception e)
            {
                Console.Error.WriteLine( "Exception: {0}", e.Message );
                exitCode = 1;
            }
            finally
            {
                // Comment the following two lines to disable waiting on exit.
                Console.Error.WriteLine( "\nPress enter to exit." );
                Console.ReadLine();
            }

            Environment.Exit( exitCode );
        }
    }
}
