// Grab.cpp
/*
	Note: Before getting started, Basler recommends reading the "Programmer's Guide" topic
	in the pylon C++ API documentation delivered with pylon.
	If you are upgrading to a higher major version of pylon, Basler also
	strongly recommends reading the "Migrating from Previous Versions" topic in the pylon C++ API documentation.

	This sample illustrates how to grab and process images using the CInstantCamera class.
	The images are grabbed and processed asynchronously, i.e.,
	while the application is processing a buffer, the acquisition of the next buffer is done
	in parallel.

	The CInstantCamera class uses a pool of buffers to retrieve image data
	from the camera device. Once a buffer is filled and ready,
	the buffer can be retrieved from the camera object for processing. The buffer
	and additional image data are collected in a grab result. The grab result is
	held by a smart pointer after retrieval. The buffer is automatically reused
	when explicitly released or when the smart pointer object is destroyed.
*/

// Include files to use the pylon API.
#include <pylon/PylonIncludes.h>
#ifdef PYLON_WIN_BUILD
#    include <pylon/PylonGUI.h>
#endif

#include <HALCONCpp/HalconCpp.h>
#include <HALCONCpp/HDevThread.h>
#include <thread>

using namespace HalconCpp;

// Namespace for using pylon objects.
using namespace Pylon;
using namespace GenApi;

using namespace HalconCpp;

// Namespace for using cout.
using namespace std;



// Number of images to be grabbed.
static const uint32_t c_countOfImagesToGrab = 100;


int main(int argc, char* argv[])
{
	// The exit code of the sample application.
	int exitCode = 0;
	SetSystem("use_window_thread", "true");

	// Before using any pylon methods, the pylon runtime must be initialized. 
	PylonInitialize();

	try
	{
		
		 // Create an instant camera object with the camera device found first.
		CInstantCamera camera(CTlFactory::GetInstance().CreateFirstDevice());
		camera.Open();
		INodeMap& nodemap = camera.GetNodeMap();
		CIntegerPtr Width = nodemap.GetNode("Width");
		CIntegerPtr Height = nodemap.GetNode("Height");
		CEnumerationPtr PixelFormat = nodemap.GetNode("PixelFormat");
		//PixelFormat->FromString("Mono8");
		//PixelFormat->FromString("BGR8");
		// PixelFormat->FromString("RGB8");
		PixelFormat->FromString("BayerBG8");
		
		HWindow window (16, 16, Width->GetValue(), Height->GetValue(), 0, "visible", "");

		HImage image;

		// Print the model name of the camera.
		cout << "Using device " << camera.GetDeviceInfo().GetModelName() << endl;

		// The parameter MaxNumBuffer can be used to control the count of buffers
		// allocated for grabbing. The default value of this parameter is 10.
		camera.MaxNumBuffer = 5;

		// Start the grabbing of c_countOfImagesToGrab images.
		// The camera device is parameterized with a default configuration which
		// sets up free-running continuous acquisition.
		camera.StartGrabbing(c_countOfImagesToGrab);

		// This smart pointer will receive the grab result data.
		CGrabResultPtr ptrGrabResult;

		CImageFormatConverter fc;
		fc.OutputPixelFormat = PixelType_BGR8packed;
		CPylonImage pylonimage;
	
		// Camera.StopGrabbing() is called automatically by the RetrieveResult() method
		// when c_countOfImagesToGrab images have been retrieved.
		while (camera.IsGrabbing())
		{
			// Wait for an image and then retrieve it. A timeout of 5000 ms is used.
			camera.RetrieveResult(5000, ptrGrabResult, TimeoutHandling_ThrowException);

			// Image grabbed successfully?
			if (ptrGrabResult->GrabSucceeded())
			{
				// Access the image data.
				cout << "SizeX: " << ptrGrabResult->GetWidth() << endl;
				cout << "SizeY: " << ptrGrabResult->GetHeight() << endl;
				const uint8_t* pImageBuffer = (uint8_t*)ptrGrabResult->GetBuffer();
				cout << "Gray value of first pixel: " << (uint32_t)pImageBuffer[0] << endl << endl;
				cout << "Gray value of first pixel: " << (uint32_t)pImageBuffer[1] << endl << endl;
				cout << "Gray value of first pixel: " << (uint32_t)pImageBuffer[2] << endl << endl;
				
				if (ptrGrabResult->GetPixelType() == EPixelType::PixelType_Mono8)
				{
					HImage img("byte", (Hlong)ptrGrabResult->GetWidth(), (Hlong)ptrGrabResult->GetHeight(), (uint8_t*)ptrGrabResult->GetBuffer());
					window.AttachBackgroundToWindow(img);
				}
				else if (ptrGrabResult->GetPixelType() == EPixelType::PixelType_BGR8packed)
				{
					
					image.GenImageInterleaved((uint8_t*)ptrGrabResult->GetBuffer(), "bgr",
						(Hlong)ptrGrabResult->GetWidth(), (Hlong)ptrGrabResult->GetHeight(),
						-1,"byte", (Hlong)ptrGrabResult->GetWidth(), (Hlong)ptrGrabResult->GetHeight(),
					     0,0,-1,0);
					window.AttachBackgroundToWindow(image);

				}
				else if (ptrGrabResult->GetPixelType() == EPixelType::PixelType_RGB8packed)
				{
					
					image.GenImageInterleaved((uint8_t*)ptrGrabResult->GetBuffer(), "rgb",
						(Hlong)ptrGrabResult->GetWidth(), (Hlong)ptrGrabResult->GetHeight(),
						-1, "byte", (Hlong)ptrGrabResult->GetWidth(), (Hlong)ptrGrabResult->GetHeight(),
						0, 0, -1, 0);
					window.AttachBackgroundToWindow(image);
				}
				else
				{
					fc.Convert(pylonimage, ptrGrabResult);
					
					image.GenImageInterleaved((uint8_t*)pylonimage.GetBuffer(), "bgr",
						(Hlong)ptrGrabResult->GetWidth(), (Hlong)ptrGrabResult->GetHeight(),
						-1, "byte", (Hlong)ptrGrabResult->GetWidth(), (Hlong)ptrGrabResult->GetHeight(),
						0, 0, -1, 0);
					window.AttachBackgroundToWindow(image);
				}	
			}
			else
			{
				cout << "Error: " << ptrGrabResult->GetErrorCode() << " " << ptrGrabResult->GetErrorDescription() << endl;
			}
		}
	}
	catch (const GenericException & e)
	{
		// Error handling.
		cerr << "An exception occurred." << endl
			<< e.GetDescription() << endl;
		exitCode = 1;
	}

	// Comment the following two lines to disable waiting on exit.
	cerr << endl << "Press enter to exit." << endl;
	while (cin.get() != '\n');

	// Releases all pylon resources. 
	PylonTerminate();

	return exitCode;
}
